
//*********************************************************************
// version I: Wed Jun 27 13:36:06 EDT 1990
//
// copyright Prof. K. Steiglitz
//          Dept. of Computer Science
//          Princeton University
//          Princeton, NJ 08544 
//
// Constraint-based design of linear-phase fir filters with
// upper and lower bounds, and convexity constraints.
// Finds minimum length, or optimizes fixed length, or pushes band-edges.
// If L is the filter length, the models are
//
// odd-length
//  cosine:   sum ( i from 0 to (L-1)/2 ) coeff[i]*cos(i*omega)
//  sine:     sum ( i from 0 to (L-3)/2 ) coeff[i]*sin((i+1)*omega)
//
// even-length
//  cosine:   sum ( i from 0 to L/2-1 ) coeff[i]*cos((i+.5)*omega)
//  sine:     sum ( i from 0 to L/2-1 ) coeff[i]*sin((i+.5)*omega) 
//*********************************************************************

#define maxpivots       1000   /* maximum no. of pivots */

#define small           1.0e-8 /* small number used in defining band-edges */
#define large           1.0e+31/* large number used in search for minimum cost column */

#define mmax            64     /* max. no. of coefficients */
#define Lmax            129    /* max. filter length, L = 2*m-1, 2*m, or 2*m+1 */
#define mmaxm           63     /* mmax - 1 */
#define nmax            640    /* max. size of n, where there are n+1 grid-points */
#define ncolmax         6000   /* max. no. of columns allowed in tableau */
#define nspecmax        20     /* max. no. of specifications */

#define eps             1.0e-8 /* for testing for zero */


Static FILE *magfile, *coefile;   /* files for output */
Static boolean done, unbounded, optimal;   /* flags for simplex */

Static enum 
{
  toomanycols, unbdual, infdual, toomanypivots, opt, infprimal
} result;   /* result of simplex */

Static long iteration;                   /* iteration count, index */
Static Char ch;                          /* character for input */
Static char m;                           /* no. of coefficients, left and right half m */
Static uchar L;                          /* filter length = 2*m-1, 2*m, 2*m+1 */
Static short n;                          /* there are n+1 grid-points from 0 to pi */
Static long numpivots;                   /* pivot count */
Static long pivotcol, pivotrow;          /* pivot column and row */
Static double pivotel;                   /* pivot element */
Static double pi;                        /* 3.14159265358979... */
Static double cbar;                      /* price when searching for entering column */
Static double coeff[mmaxm + 1];          /* coefficients */
Static double carry[mmax + 2][mmax + 2]; /* inverse-basis matrix of the
                                              revised simplex method */
Static char phase;                       /* phase */
Static double price[mmax + 1];           /* shadow prices = row -1 of carry =
                                            -dual variables = -coefficients */
Static long basis[mmax + 1];             /* basis columns, negative integers
                                            artificial */
Static char nspec;   /* no. of bands */
Static enum 
{
  con, lim
} spectype[nspecmax];                             /* type of band */
Static double left[nspecmax], right[nspecmax];    /* bandedges as read in */
Static double freq[ncolmax];                      /* frequencies at grid points */
Static double bound1[nspecmax], bound2[nspecmax]; /* left and right bounds */
Static Char sense[nspecmax];                      /* sense of constraint, + up, - down */
Static Char interpolate[nspecmax];                /* g=geometric, a=arithmetic */
Static short firstcol[nspecmax];                  /* leftmost column of spec */
Static short lastcol[nspecmax];                   /* rightmost column of spec */
Static boolean foundfeas;                         /* found feasible solution */
Static long mlargest, msmallest;                  /* range of m */
Static long Llargest, Lsmallest;                  /* range of L = 2*m-1, 2*m, or 2*m+1 */
Static short ncol;                                /* number of columns */
Static double tab[mmax + 1][ncolmax];             /* tableau */
Static double d[ncolmax];                         /* current cost vector */
Static double c[ncolmax];                         /* cost in original problem */
Static double curcol[mmax + 2];                   /* current column */
Static double curcost;                            /* current cost */
Static long bestm;                                /* best order */
Static Char hug[nspecmax];                        /* allow this constraint to be hugged? */
Static enum 
{
  findlen, maxdist, pushedge
} whattodo;   /* type of optimization */
Static long npushed;                              /* number of bandedges pushed */
Static enum 
{
  rr, ll
} whichway;   /* push which way? */
Static long bandpushed[nspecmax];                 /*  bandedges pushed */
Static double lowlim;                             /* lower limit for finding if primal is feasible */
Static boolean oddlength;                         /* odd-length filters? */
Static enum 
{
  cosine, sine
} symtype;   /* cosine or sine symmetry */


Static Void makebands(long i)
//*******************************************************************
// fills in frequencies to make grid - frequencies are kept as reals
// in radians, and each band has equally spaced grid points          
//*******************************************************************
{
  long k, kmax;

  if (i == 1)
  {
    firstcol[i - 1] = 1;
  }
  else
  {
    firstcol[i - 1] = lastcol[i - 2] + 1;
  }
  kmax = (long)((right[i - 1] - left[i - 1]) * n / 0.5 + small);/* kmax+1 cols. in this band */

  if (kmax == 0)
  {
    freq[firstcol[i - 1] - 1] = 2.0 * pi * left[i - 1];
  }
  else 
  {
    for (k = 0; k <= kmax; k++)
	{
      freq[firstcol[i - 1] + k - 1] = 2.0 * pi * (left[i - 1] + (right[i - 1] - left[i - 1]) * k / kmax);
	}
  }
  lastcol[i - 1] = firstcol[i - 1] + kmax;
}  /* makebands */


Static double trig0(long i, double f)
//*******************************************************************
// trig function in filter transfer function
//           
//*******************************************************************
{
  double Result;

  if (oddlength) 
  {
    if (symtype == cosine)
	{
      Result = cos(i * f);
	}
    else
	{
      Result = sin((i + 1) * f);
	}
  }

  if (oddlength)
  {
    return Result;
  }
  if (symtype == cosine)
  {
    return cos((i + 0.5) * f);
  }
  else 
  {
    return sin((i + 0.5) * f);
  }
  return Result;  /* remove to avoid warning */ 
}  /* trig0 */


Static double trig2(long i, double f)
//*******************************************************************
// second derivative of trig function in filter transfer function
//           
//*******************************************************************
{
  double Result;

  if (oddlength) 
  {
    if (symtype == cosine)
	{
      Result = -i * i * cos(i * f);
	}
    else
	{
      Result = -(i + 1) * (i + 1) * sin((i + 1) * f);
	}
  }

  if (oddlength)
  {
    return Result;
  }
  if (symtype == cosine)
  {
    return (-(i + 0.5) * (i + 0.5) * cos((i + 0.5) * f));
  }
  else 
  {
    return (-(i + 0.5) * (i + 0.5) * sin((i + 0.5) * f));
  }
  /* return Result; removed to avoid warning */
}  /* trig2 */


Static Void convex(long i)
//*******************************************************************
// sets up tableau columns for convexity constraints on magnitude
//           
//*******************************************************************
{
  long row, col;   /* gridpoint, side of polygon, row, col */
  long FORLIM, FORLIM1;

  makebands(i);
  FORLIM = lastcol[i - 1];
  for (col = firstcol[i - 1] - 1; col < FORLIM; col++)
  {   /* for all frequencies in band */
    c[col]  = 0.0;
    FORLIM1 = m;
    for (row = 0; row < FORLIM1; row++) 
    {
      tab[row][col] = trig2(row, freq[col]);   /* normal constraint is <= */
      if (sense[i - 1] == '+')
	  {
	    tab[row][col] = -tab[row][col];
	  }
    }  /* for row */
    tab[m][col] = 0.0;
  }  /*for col */
}  /* convex */


Static Void limit(long i)
//*******************************************************************
// 
// sets up tableau columns for upper or lower bounds on transfer
// function for specification i; the bound is linearly interpolated
// between the start and end of the band. 
//           
//*******************************************************************
{
  long row, col;   /* gridpoint, side of polygon, row, col */
  long FORLIM, FORLIM1;

  makebands(i);
  FORLIM = lastcol[i - 1];
  for (col = firstcol[i - 1] - 1; col < FORLIM; col++)
  {   /* for all frequencies in band */
    if (firstcol[i - 1] == lastcol[i - 1])
	{
      c[col] = bound1[i - 1];
	}
    else 
    {
      if (interpolate[i - 1] == 'g')
	  {
	    c[col] = bound1[i - 1] * exp((double)(col - firstcol[i - 1] + 1) /
				     (lastcol[i - 1] - firstcol[i - 1]) * log(
				       fabs(bound2[i - 1] / bound1[i - 1])));
	  }
      if (interpolate[i - 1] == 'a')
	  {
	    c[col] = bound1[i - 1] + (double)(col - firstcol[i - 1] + 1) /
				 (lastcol[i - 1] - firstcol[i - 1]) *
				 (bound2[i - 1] - bound1[i - 1]);
	  }
    }
    if (sense[i - 1] == '-')
	{
      c[col] = -c[col];
	}
    FORLIM1 = m;
    for (row = 0; row < FORLIM1; row++) 
    {
      tab[row][col] = trig0(row, freq[col]);
      if (sense[i - 1] == '-')
	  {
	    tab[row][col] = -tab[row][col];
	  }
    }
    if (hug[i - 1] != 'h')
	{
      tab[m][col] = 1.0;
	}
    else
	{
      tab[m][col] = 0.0;
	}
  }  /* for col */
}  /* limit */


Static Void readdata()
//*******************************************************************
// 
// reads in problem data
// not meant to be interactive; aborts on bad filter lengths
// 
//           
//*******************************************************************
{
  long i;
  boolean allhugged;   /* to see if all constraints are hugged */
  long FORLIM;

  scanf("%ld%ld%*[^\n]", &Lsmallest, &Llargest);
  getchar();

  if (Lsmallest < 1 || Llargest > Lmax) 
  {
    System.out.println("Lsmallest or Llargest out of range: quitting");
    exit(0);
  }

  if ((Lsmallest & 1) != (Llargest & 1)) 
  {
    System.out.println("parity of Lsmallest and Llargest unequal: quitting");
    exit(0);
  }

  scanf("%c%*[^\n]", &ch);
  getchar();
  if (ch == '\n')
  {
    ch = ' ';
  }
  if (ch == 'c')
  {
    symtype = cosine;
  }
  else
  {
    symtype = sine;
  }
  if (symtype == cosine)
  {
    System.out.println("cosine symmetry");
  }
  else
  {
    System.out.println("sine symmetry");
  }

  oddlength = Lsmallest & 1;

  if (oddlength) 
  {
    if (symtype == cosine) 
    {
      msmallest = (Lsmallest + 1) / 2;
      mlargest  = (Llargest + 1) / 2;
    } 
    else 
    {
      msmallest = (Lsmallest - 1) / 2;
      mlargest  = (Llargest - 1) / 2;
    }
  }

  if (!oddlength) 
  {
    msmallest = Lsmallest / 2;
    mlargest  = Llargest / 2;
  }

  if (Lsmallest != Llargest) 
  {
    whattodo = findlen;
    System.out.println("finding minimum length: range "+Lsmallest+" to "+Llargest);
  }

  if (Lsmallest == Llargest) 
  {
    m = msmallest;
    L = Lsmallest;
    System.out.println("fixed length of"+ L);
    scanf("%c%*[^\n]", &ch);
    getchar();   /* right, left, or neither: edges to be pushed? */
    if (ch == '\n')
	{
      ch = ' ';
	}
    if (ch == 'n') 
    {
      whattodo = maxdist;
      System.out.println("maximizing distance from constraints not marked hugged");
    } 
    else 
    {
      whattodo = pushedge;
      if (ch == 'r')
	  {
	    whichway = rr;
	  }
      else
	  {
	    whichway = ll;
	  }
      scanf("%ld%*[^\n]", &npushed);
      getchar();
      FORLIM = npushed;
      for (i = 0; i < FORLIM; i++)
	  {
	    scanf("%ld", &bandpushed[i]);
	  }
      scanf("%*[^\n]");
      getchar();
      if (whichway == rr)
	  {
	    System.out.println("pushing bandedges right");
	  }
      if (whichway == ll)
	  {
	    System.out.println("pushing bandedges left");
	  }
      System.out.println("constraint numbers: ");
      FORLIM = npushed;
      for (i = 0; i < FORLIM; i++)
	  {
	    System.out.println(bandpushed[i]);
	  }
      putchar('\n');
    }
  }
  scanf("%hd%*[^\n]", &n);
  getchar();   /* there are n+1 grid points between 0 and pi */
  nspec = 0;
  scanf("%c%*[^\n]", &ch);
  getchar();
  if (ch == '\n')
  {
    ch = ' ';
  }
  while (ch != 'e') /* 'e' for end */
  {   
    nspec++;
    i = nspec;
    if (ch == 'c') 
    {
      spectype[i - 1] = con;
      scanf("%c%*[^\n]", &sense[i - 1]);
      getchar();
      if (sense[i - 1] == '\n')
	  {
	    sense[i - 1] = ' ';
	  }
      scanf("%lg%lg%*[^\n]", &left[i - 1], &right[i - 1]);
      getchar();
      System.out.println("constraint "+i+": convex, sense "+sense[i - 1]);
      System.out.println("  bandedges:  "+left[i - 1]+" "+right[i - 1]);
    }
    if (ch == 'l') 
    {
      spectype[i - 1] = lim;
      scanf("%c%*[^\n]", &sense[i - 1]);
      getchar();
      if (sense[i - 1] == '\n')
	  {
	    sense[i - 1] = ' ';
	  }
      scanf("%c%*[^\n]", &interpolate[i - 1]);
      getchar();
      if (interpolate[i - 1] == '\n')
	  {
	    interpolate[i - 1] = ' ';
	  }
      scanf("%c%*[^\n]", &hug[i - 1]);
      getchar();
      if (hug[i - 1] == '\n')
	  {
	    hug[i - 1] = ' ';
	  }
      scanf("%lg%lg%*[^\n]", &left[i - 1], &right[i - 1]);
      getchar();
      scanf("%lg%lg%*[^\n]", &bound1[i - 1], &bound2[i - 1]);
      getchar();
      if (interpolate[i - 1] == 'g' && bound1[i - 1] * bound2[i - 1] == 0.0) 
      {
	    System.out.println("geometrically interpolated bandedge in constraint "+i+" is zero");
	    exit(0);
      }
      if (sense[i - 1] == '+')
	  {
	    System.out.println("constraint "+i+": upper limit");
	  }
      if (sense[i - 1] == '-')
	  {
	    System.out.println("constraint "+i+": lower limit\n");
	  }
      if (interpolate[i - 1] == 'g')
	  {
	    System.out.println("  geometric interpolation");
	  }
      if (interpolate[i - 1] == 'a')
	  {
	    System.out.println("  arithmetic interpolation");
	  }
      if (hug[i - 1] == 'h')
	  {
	    System.out.println("  this constraint will be hugged");
	  }
      else
	  {
	    System.out.println("  this constraint will be optimized");
	  }
      System.out.println("  bandedges:  "+left[i - 1]+" "+ right[i - 1]);
      System.out.println("  bounds:     "+bound1[i - 1]+" "+ bound2[i - 1]);
    }
    makebands(i);
    System.out.println("  initial columns:    "+ firstcol[i - 1]+" "+ lastcol[i - 1]);
    scanf("%c%*[^\n]", &ch);
    getchar();   /* next */
    if (ch == '\n')
	{
      ch = ' ';
	}
  }  /* while */
  ncol = lastcol[nspec - 1];
  System.out.println("number of specs= "+ nspec);
  System.out.println("initial number of columns= "+ ncol);

  allhugged = true;   /* check to see if all limit constraints are hugged */
  FORLIM = nspec;
  for (i = 0; i < FORLIM; i++) 
  {
    if (spectype[i] == lim && hug[i] != 'h')
	{
      allhugged = false;
	}
  }
  if (allhugged) 
  {
    System.out.println("all constraints are hugged: ill-posed problem");
    exit(0);
  }

}  /* readdata */


Static Void setup()
//*******************************************************************
// 
// initializes constraints
//
// 
//           
//*******************************************************************
{
  long i, FORLIM;

  pi = 4.0 * atan(1.0);
  FORLIM = nspec;
  for (i = 1; i <= FORLIM; i++) 
  {
    switch (spectype[i - 1]) 
    {

    case con:
      convex(i);
      break;

    case lim:
      limit(i);
      break;
    }/* case */
  }  /*for */
  ncol = lastcol[nspec - 1];
}  /* setup */


Static Void columnsearch()
//*******************************************************************
// 
// looks for favorable column to enter basis.
// returns lowest cost and its column number, or turns on the flag optimal
// 
//           
//*******************************************************************
{
  long i, col;
  double tempcost;   /* minimum cost, temporary cost of column */
  long FORLIM, FORLIM1;

  FORLIM = m;
  for (i = 0; i <= FORLIM; i++)   /* set up price vector */
  {
    price[i] = -carry[0][i + 1];
  }
  optimal  = false;
  cbar     = large;
  pivotcol = 0;
  FORLIM   = ncol;
  for (col = 1; col <= FORLIM; col++) 
  {
    tempcost = d[col - 1];
    FORLIM1 = m;
    for (i = 0; i <= FORLIM1; i++)
	{
      tempcost -= price[i] * tab[i][col - 1];
	}
    if (cbar > tempcost) 
    {
      cbar = tempcost;
      pivotcol = col;
    }
  }  /* for col */
  if (cbar > -eps)
  {
    optimal = true;
  }
}  /* columnsearch */


Static Void rowsearch()
//*******************************************************************
// 
// looks for pivot row. returns pivot row number,
// or turns on the flag unbounded 
// 
//           
//*******************************************************************
{
  long i, j;
  double ratio, minratio;   /* ratio and minimum ratio for ratio test */
  long FORLIM, FORLIM1;

  FORLIM = m + 1;
  for (i = 1; i <= FORLIM; i++) /* generate column */
  {   
    curcol[i] = 0.0;   /* current column = B inverse * original col. */
    FORLIM1 = m;
    for (j = 0; j <= FORLIM1; j++)
	{
      curcol[i] += carry[i][j + 1] * tab[j][pivotcol - 1];
	}
  }
  curcol[0] = cbar;   /* first element in current column */
  pivotrow  = -1;
  minratio  = large;
  FORLIM    = m;
  for (i = 0; i <= FORLIM; i++) /* ratio test */
  {   
    if (curcol[i + 1] > eps) 
    {
      ratio = carry[i + 1][0] / curcol[i + 1];
      if (minratio > ratio) /* favorable row */
      {   
	    minratio = ratio;
	    pivotrow = i;
	    pivotel  = curcol[i + 1];
      } 
      else 	/* break tie with max pivot */
      {  
	    if (minratio == ratio && pivotel < curcol[i + 1]) 
	    {
	      pivotrow = i;
	      pivotel = curcol[i + 1];
	    }
      }
    }  /* curcol > eps */
  }  /* for i */
  if (pivotrow == -1)
  {
    unbounded = true;   /* nothing found */
  }
  else
  {
    unbounded = false;
  }
}  /* rowsearch */


Static Void pivot()
//*******************************************************************
// pivots
//           
//*******************************************************************
{
  /* pivots */
  long i, j, FORLIM, FORLIM1;

  basis[pivotrow] = pivotcol;
  FORLIM = m + 1;
  for (j = 0; j <= FORLIM; j++)
  {
    carry[pivotrow + 1][j] /= pivotel;
  }
  FORLIM = m + 1;
  for (i = 0; i <= FORLIM; i++) 
  {
    if (i - 1 != pivotrow) 
    {
      FORLIM1 = m + 1;
      for (j = 0; j <= FORLIM1; j++)
	  {
	    carry[i][j] -= carry[pivotrow + 1][j] * curcol[i];
	  }
    }
  }
  curcost = -carry[0][0];
}  /* pivot */


Static Void changephase()
//*******************************************************************
// 
// changes phase from 1 to 2, by switching to original cost vector
//           
//*******************************************************************
{
  long i, j, b, FORLIM, FORLIM1;

  phase = 2;
  FORLIM = m;
  for (i = 0; i <= FORLIM; i++) 
  {
    if (basis[i] <= 0)
	{
      System.out.println("...artificial basis element "+basis[i]+" remains in basis after phase 1");
	}
  }
  FORLIM = ncol;
  for (j = 0; j < FORLIM; j++)   /* switch to original cost vector */
  {
    d[j] = c[j];
  }
  FORLIM = m + 1;
  for (j = 0; j <= FORLIM; j++) 
  {
    carry[0][j] = 0.0;
    FORLIM1 = m;
    for (i = 0; i <= FORLIM1; i++) 
    {
      b = basis[i];   /* ignore artificial basis elements that are */
      if (b >= 1)   /* still in basis */
	  carry[0][j] -= c[b - 1] * carry[i + 1][j];
    }  /* for i */
  }  /* for j */
  curcost = -carry[0][0];
}  /* changephase */


Static double mag(double f)
//*******************************************************************
// 
// computes magnitude function, given radian frequency f 
//           
//*******************************************************************
{
  long i;
  double temp;
  long FORLIM;

  temp = 0.0;
  FORLIM = m;
  for (i = 0; i < FORLIM; i++)
  {
    temp += coeff[i] * trig0(i, f);
  }
  return temp;
}  /* mag */


Static Void wrapup()
//*******************************************************************
// 
// prints results, final frequency response 
//           
//*******************************************************************
{
  long i, k, FORLIM;

  if (magfile != NULL)
  {
    magfile = freopen("magfile", "wb", magfile);
  }
  else
  {
    magfile = fopen("magfile", "wb");
  }
  if (magfile == NULL)
  {
    exit(FileNotFound);
  }
  if (coefile != NULL) 
  {
    /* open files for writing */
    coefile = freopen("coefile", "wb", coefile);
  } 
  else
  {
    coefile = fopen("coefile", "wb");
  }
  if (coefile == NULL)
  {
    exit(FileNotFound);
  }

  if (oddlength && symtype == cosine) /* write coeffs to coefile */
  {   
    fprintf(coefile, "%4d cosine symmetry\n", m * 2 - 1);
	/* L = length, odd */
    for (i = m - 1; i >= 1; i--)
	{
      fprintf(coefile, "% .5E\n", coeff[i] / 2);
	}
    fprintf(coefile, "% .5E\n", coeff[0]);
    FORLIM = m;
    for (i = 1; i < FORLIM; i++)
	{
      fprintf(coefile, "% .5E\n", coeff[i] / 2);
	}
  }  /* odd, cosine */

  if (!oddlength && symtype == cosine) 
  {
    fprintf(coefile, "%4d cosine symmetry\n", m * 2);   /* L = length, even */
    for (i = m - 1; i >= 0; i--)
	{
      fprintf(coefile, "% .5E\n", coeff[i] / 2);
	}
    FORLIM = m;
    for (i = 0; i < FORLIM; i++)
	{
      fprintf(coefile, "% .5E\n", coeff[i] / 2);
	}
  }  /* even, cosine */

  if (oddlength && symtype == sine) 
  {
    fprintf(coefile, "%4d sine symmetry\n", m * 2 + 1);
	/* L = length, odd */
    for (i = m - 1; i >= 0; i--)   /* negative of the first m coefs. */
	{
      fprintf(coefile, "% .5E\n", coeff[i] / -2);
	}
    fprintf(coefile, " 0.0\n");   /* middle coefficient is always 0 */
    FORLIM = m;
    for (i = 0; i < FORLIM; i++)
	{
      fprintf(coefile, "% .5E\n", coeff[i] / 2);
	}
  }  /* odd, sine */

  if (!oddlength && symtype == sine) 
  {
    fprintf(coefile, "%4d sine symmetry\n", m * 2);   /* L = length, even */
    for (i = m - 1; i >= 0; i--)   /* negative of the first m coefs. */
	{
      fprintf(coefile, "% .5E\n", coeff[i] / -2);
	}
    FORLIM = m;
    for (i = 0; i < FORLIM; i++)
	{
      fprintf(coefile, "% .5E\n", coeff[i] / 2);
	}
  }  /* even, sine */

  FORLIM = n;
  for (k = 0; k <= FORLIM; k++)   /* magnitude on regular grid */
  {
    fprintf(magfile, "%10.4f % .5E\n", 0.5 * k / n, mag(k * pi / n));
  }
  fprintf(magfile, "\n       magnitude at bandedges\n\n");
  FORLIM = nspec;
  for (i = 0; i < FORLIM; i++) 
  {
    if (spectype[i] == lim) 
    {
      fprintf(magfile, "%10.4f % .5E\n", freq[firstcol[i] - 1] * 0.5 / pi, mag(freq[firstcol[i] - 1]));
      fprintf(magfile, "%10.4f % .5E\n", freq[lastcol[i] - 1] * 0.5 / pi, mag(freq[lastcol[i] - 1]));
      putchar('\n');
    }
  }
}  /* wrapup */


Static Void simplex()
//*******************************************************************
// 
// simplex for linear programming 
//           
//*******************************************************************
{
  long i, j, col, row, FORLIM, FORLIM1;

  done = false;
  phase = 1;
  FORLIM = m + 1;
  for (i = 0; i <= FORLIM; i++) 
  {
    for (j = 0; j <= mmax + 1; j++)
	{
      carry[i][j] = 0.0;
	}
  }
  FORLIM = m + 1;
  for (i = 1; i <= FORLIM; i++)   /* artificial basis */
  {
    carry[i][i] = 1.0;
  }
  carry[0][0] = -1.0;   /* - initial cost */
  curcost = -carry[0][0];
  carry[m + 1][0] = 1.0;   /* variable minimized in primal */
  FORLIM = m;
  for (i = 0; i <= FORLIM; i++)   /* initial, artificial basis */
  {
    basis[i] = -i;
  }
  if (ncol <= ncolmax) /* check number of columns */
  {   
    FORLIM = ncol;
    for (col = 0; col < FORLIM; col++) /* initialize cost for phase 1 */
    {   
      d[col] = 0.0;
      FORLIM1 = m;
      for (row = 0; row <= FORLIM1; row++)
	  {
	    d[col] -= tab[row][col];
	  }
    }
  } 
  else 
  {
    System.out.println("...termination: too many columns for storage");
    done = true;
    result = toomanycols;
  }
  numpivots = 0;
  while (numpivots < maxpivots && !done && (curcost > lowlim || phase == 1)) 
  {
    columnsearch();
    if (!optimal)  /* not optimal */
    { 
      rowsearch();
      if (unbounded) 
      {
	    done = true;
	    result = unbdual;   /* dual of problem is unbounded */
      } 
      else 
      {
	    pivot();
	    numpivots++;
	    if (numpivots == 1 || numpivots % 10 == 0)
		{
	      System.out.println("pivot "+numpivots+" cost= "+ curcost);
/* p2c: meteorJ.p, line 595:
 * Note: Using % for possibly-negative arguments [317] */
        }
      }
      continue;
    }  /* not optimal */
    if (phase == 1) 
    {
      if (curcost > eps) 
      {
	    done = true;   /* dual of problem is infeasible */
	    result = infdual;   /* this happens if all specs are hugged */
      } 
      else 
      {
	    if (numpivots != 1 && numpivots % 10 != 0)
		{
	      System.out.println("pivot "+numpivots+" cost= "+ curcost);
/* p2c: meteorJ.p, line 609:
 * Note: Using % for possibly-negative arguments [317] */
        }
	    System.out.println("phase 1 successfully completed\n");
	    changephase();
      }
      continue;
    }  /* if phase = 1 */
    if (numpivots != 1 && numpivots % 10 != 0)
	{
	   System.out.println("pivot "+numpivots+" cost= "+ curcost);
/* p2c: meteorJ.p, line 617:
 * Note: Using % for possibly-negative arguments [317] */
    }
    System.out.println("phase 2 successfully completed");
    done = true;
    result = opt;
  }  /* while */
  if (curcost <= lowlim && phase == 2) 
  {
    if (numpivots != 1 && numpivots % 10 != 0)
	{
	   System.out.println("pivot "+numpivots+" cost= "+ curcost);
/* p2c: meteorJ.p, line 626:
 * Note: Using % for possibly-negative arguments [317] */
    }
    result = infprimal;
  }
  if (numpivots >= maxpivots) 
  {
    System.out.println("...termination: maximum number of pivots exceeded");
    result = toomanypivots;
  }

  /* optimal */
}  /* simplex */


Static Void printresult()
//*******************************************************************
// 
// prints enumerated type 
//           
//*******************************************************************
{
  switch (result) 
  {

  case toomanycols:
    System.out.println("too many columns in specifications");
    break;

  case unbdual:   /* unbounded dual */
    System.out.println("infeasible (unbounded dual)");
    break;

  case infdual:   /* infeasible dual */
    System.out.println("infeasible or unbounded");
    break;

  case toomanypivots:
    System.out.println("too many pivots");
    break;

  case opt:
    System.out.println("optimum obtained");
    break;

  case infprimal:
    System.out.println("infeasible ");
    break;
  }/* case */
}  /* printresult */


Static Void getm()
//*******************************************************************
// 
// find best order (and hence length) 
//           
//*******************************************************************
{
  char leftm, rightm;
  long i;
  boolean foundm, checkedleft, checkedright;
  long FORLIM;

  foundfeas = false;   /* flag set when a feasible solution is found */
  iteration = 0;
  leftm     = msmallest;
  rightm    = mlargest;
  foundm    = false;
  checkedleft = false;
  checkedright = false;
  while (!foundm) 
  {
    if (iteration == 0)   /* first time through */
	{
      m = leftm + (rightm - leftm) / 2;
	}
    iteration++;
    System.out.println("iteration "+ iteration);

    if (oddlength) 
    {
      if (symtype == cosine)
	  {
	    System.out.println("L= "+ (m * 2 - 1));
	  }
      else
	  {
	    System.out.println("L= "+ (m * 2 + 1));
	  }
    }

    if (!oddlength)
	{
      System.out.println("L= "+ (m * 2));
	}

    setup();
    simplex();
    printresult();
    if (result == opt) 
    {
      foundfeas = true;
      rightm    = m;
      bestm     = m;
      checkedright = true;   /* right side of bracket has been checked */
      if (oddlength) 
      {
	    if (symtype == cosine)
		{
	      System.out.println("new best length L= "+ (bestm * 2 - 1));
		}
      	else
		{
	      System.out.println("new best length L= "+ (bestm * 2 + 1));
		}
      }

      if (!oddlength)
	  {
	    System.out.println("new best length L= "+ (bestm * 2));
	  }

      FORLIM = m;
      for (i = 0; i < FORLIM; i++)
	  {
	    coeff[i] = -carry[0][i + 1];
	  }
    }

    if (result != opt) 
    {
      leftm = m;
      checkedleft = true;   /* left side of bracket has been checked */
    }

    if (rightm > leftm + 1)
	{
      m = leftm + (rightm - leftm) / 2;
	}

    if (rightm == leftm + 1) 
    {
      if (!checkedleft) 
      {
    	m = leftm;
	    checkedleft = true;
      } 
      else if (!checkedright) 
      {
    	m = rightm;
	    checkedright = true;
      } 
      else
	  {
	    foundm = true;
	  }
    }

    if (rightm == leftm) 
    {
      foundm = true;
    }
  }  /* while */

  if (!foundfeas) 
  {
    System.out.println("no feasible solution found");
    exit(0);
  }
  m = bestm;

  putchar('\n');
  if (oddlength) 
  {
    if (symtype == cosine)
	{
      System.out.println("best length L= "+ (bestm * 2 - 1));
	}
    else
	{
      System.out.println("best length L= "+ (bestm * 2 + 1));
	}
  }

  if (!oddlength) 
  {
    System.out.println("best length L= "+ (bestm * 2));
  }
}  /* getm */


Static Void getedge()
//*******************************************************************
// 
// optimize bandedge 
//           
//*******************************************************************
{
  double lefte, righte, newe, beste, onespace, stopspace;
      /* nominal grid spacing, stop criterion */
  long i, FORLIM;

  onespace = 0.5 / n;   /* space between grid points */
  stopspace = onespace / 10.0;
      /* stop criterion is 1/10 nominal grid spacing */
  if (whichway == rr) 
  {
    lefte = left[bandpushed[0] - 1];   /* start with rightmost leftedge */
    FORLIM = npushed;
    for (i = 1; i < FORLIM; i++) 
    {
      if (left[bandpushed[i] - 1] > lefte)
	  {
	     lefte = left[bandpushed[i] - 1];
	  }
    }
    righte = 0.5;
  } 
  else 
  {
    lefte  = 0.0;
    righte = right[bandpushed[0] - 1];
    FORLIM = npushed;
    for (i = 1; i < FORLIM; i++) /* start with leftmost rightedge */
    {   
      if (right[bandpushed[i] - 1] < righte)
	  {
	     righte = right[bandpushed[i] - 1];
	  }
    }
  }
  iteration = 0;
  foundfeas = false;   /* flag set when a feasible solution is found */
  while (righte - lefte > stopspace) 
  {
    iteration++;
    newe = (righte + lefte) / 2.0;
    System.out.println("iteration "+ iteration);
    System.out.println("trying new edge = "+ newe);
    FORLIM = npushed;
    for (i = 0; i < FORLIM; i++) 
    {
      if (whichway == rr)
	  {
	    right[bandpushed[i] - 1] = newe;
	  }
      else
	  {
	    left[bandpushed[i] - 1] = newe;
	  }
    }
    setup();
    simplex();
    printresult();
    if (result == opt) 
    {
      if (whichway == rr)
	  {
	    lefte = newe;
	  }
      else
	  {
	    righte = newe;
	  }
      foundfeas = true;
      beste = newe;
      FORLIM = m;
      for (i = 0; i < FORLIM; i++)
	  {
	    coeff[i] = -carry[0][i + 1];
	  }
    }
    if (result != opt) 
    {
      if (whichway == rr)
	  {
	    righte = newe;
	  }
      else
	  {
	    lefte = newe;
	  }
    }
  }  /* while */
  putchar('\n');
  if (!foundfeas) 
  {
    System.out.println("no feasible bandedge found");
    exit(0);
  }
  System.out.println("found edge= "+ beste);
  FORLIM = npushed;
  for (i = 0; i < FORLIM; i++) 
  {
    if (whichway == rr)
	{
      right[bandpushed[i] - 1] = beste;
	}
    else
	{
      left[bandpushed[i] - 1] = beste;
	}
  }
  FORLIM = nspec;
  for (i = 1; i <= FORLIM; i++)
  {
    makebands(i);
  }
}  /* getedge */


Static Void getmaxdist()
//*******************************************************************
// 
// maximizes distance from constraints 
//           
//*******************************************************************
{
  long i, FORLIM;

  System.out.println("optimization: maximize distance from constraints");
  setup();
  simplex();
  printresult();
  if (result != opt)
  {
    exit(0);
  }
  System.out.println("final cost= distance from constraints= "+ curcost);
  FORLIM = m;
  for (i = 0; i < FORLIM; i++)   /* record coefficients */
  {
    coeff[i] = -carry[0][i + 1];
  }
  /* don't go back if unsuccessful */
}  /* getmaxdist */


main(int argc, char *argv[])
//*******************************************************************
// 
// Main program 
//           
//*******************************************************************
{  
//  PASCAL_MAIN(argc, argv);
  coefile = NULL;
  magfile = NULL;
  System.out.println("welcome to meteor:");
  System.out.println("constraint-based, linear-phase FIR filter design");
  readdata();
  lowlim = -eps;   /* dual cost negative => primal infeasible */
  switch (whattodo) /* case */
  {   
  case findlen:
    getm();
    break;

  case pushedge:
    getedge();
    break;

  case maxdist:
    getmaxdist();
    break;
  }
  /* no return here if unsuccessful */
  wrapup();
  if (magfile != NULL)
  {
    fclose(magfile);
  }
  if (coefile != NULL)
  {
    fclose(coefile);
  }
  exit(0);
}  /* main */




/* End. */
